const system = require("../../../system");
const ServiceBase = require("../../sve.base");
const settings = require("../../../../config/settings");

class TradeMarkService extends ServiceBase {
  constructor() {
    super("dbtrademark", ServiceBase.getDaoName(TradeMarkService));
    this.ordertmproductSve = system.getObject("service.dborder.ordertmproductSve");
  }

  /**
   * 修改商标信息 （订单详情页面使用）
   * @param {*} obj 
   * obj.deliveryOrderNo 交付订单号，
   * obj.tmName  商标名称，
   * obj.tmFormType 商标类型，
   * obj.notes 商标说明，
   * obj.picUrl 商标图样,
   * obj.colorizedPicUrl 商标彩色图样,
   * obj.user 用户数据
   */
  async updateTmInfo(obj){
    var user = obj.user;
    if(!user || !user.id){
      return system.getResultFail(-100, "未知用户");
    }
    var deliveryOrderNo = obj.deliveryOrderNo;
    if(!deliveryOrderNo){
      return system.getResultFail(-101, "deliveryOrderNo参数错误");
    }
    // 1.获取交付单信息
    var ordertmproduct = await this.ordertmproductSve.dao.model.findOne({
      where:{deliveryOrderNo:deliveryOrderNo},
      raw:true
    });
    if(!ordertmproduct || !ordertmproduct.id){
      return system.getResultFail(-102, "商标交付单不存在");
    }
    // 2.获取交付单状态，判断是否可修改
    if(ordertmproduct.deliveryStatus=='ddj' || ordertmproduct.deliveryStatus=='ywc'){
      var deliveryStatusName = "待递交";
      if(ordertmproduct.deliveryStatus=='ywc'){
        deliveryStatusName="已完成";
      }
      return system.getResultFail(-103, "该商标交付单状态为"+deliveryStatusName+",不能进行修改");
    }
    var self = this;
    return await self.db.transaction(async function (t) {
      var otpObj={
        id:ordertmproduct.id,
        deliveryOrderNo:deliveryOrderNo,
        updateuser_id:user.id,
        updateuser:user.nickname
      };
      if(obj.picUrl){//商标图样 黑白
        otpObj["picUrl"]=obj.picUrl;
      }
      if(obj.colorizedPicUrl){//彩色商标图样
        otpObj["colorizedPicUrl"]=obj.colorizedPicUrl;
      }
      if(obj.tmName){//商标名称
        otpObj["tmName"]=obj.tmName;
      }
      if(obj.tmFormType){//商标类型
        otpObj["tmFormType"]=obj.tmFormType;
      }
      if(obj.notes){//商标说明
        otpObj["notes"]=obj.notes;
      }
      await self.ordertmproductSve.dao.update(otpObj,t);//商标交付单 修改商标图样
      return system.getResultSuccess();
    })
  }

  /**
   * 修改商标尼斯信息 （订单详情页面使用）
   * @param {*} obj 
   * obj.tbCode 商标提报号，
   * obj.deliveryOrderNo 交付订单号，
   * obj.nclOneCodes 商标尼斯大类，
   * obj.nclSmallCodes 商标尼斯小类数组
   * obj.user 用户数据
   */
  async updateNclInfo(obj){
    var user = obj.user;
    var self = this;
    if(!user || !user.id){
      return system.getResultFail(-100, "未知用户");
    }
    var deliveryOrderNo = obj.deliveryOrderNo;
    if(!deliveryOrderNo){
      return system.getResultFail(-101, "deliveryOrderNo参数错误");
    }
    // 1.获取交付单信息
    var ordertmproduct = await this.ordertmproductSve.dao.model.findOne({
      where:{deliveryOrderNo:deliveryOrderNo},
      raw:true
    });
    if(!ordertmproduct || !ordertmproduct.id){
      return system.getResultFail(-102, "商标交付单不存在");
    }
    // 2.获取交付单状态，判断是否可修改
    if(ordertmproduct.deliveryStatus=='ddj' || ordertmproduct.deliveryStatus=='ywc'){
      var deliveryStatusName = "待递交";
      if(ordertmproduct.deliveryStatus=='ywc'){
        deliveryStatusName="已完成";
      }
      return system.getResultFail(-103, "该商标交付单状态为"+deliveryStatusName+",不能进行修改");
    }
    var tbCode = obj.tbCode;
    if(!tbCode){
      return system.getResultFail(-104, "tbCode参数错误");
    }
    //获取商标尼斯信息
    var tm = await this.dao.model.findOne({
      where:{tbCode:tbCode},
      raw:true
    });
    
    if(!tm || !tm.id){
      return system.getResultFail(-105, "尼斯信息不存在");
    }
    //获取交付单下其它商标尼斯信息
    var othertm = await this.dao.model.findAll({
      where:{
        deliveryOrderNo:deliveryOrderNo,
        tbCode: { [self.db.Op.ne]: tbCode } 
      },
      raw:true
    });
    if(!obj.nclOneCodes){
      return system.getResultFail(-106, "nclOneCodes参数错误");
    }
    if(!obj.nclSmallCodes || obj.nclSmallCodes.length<1){
      return system.getResultFail(-107, "nclSmallCodes参数错误");
    }
    if(obj.nclSmallCodes.length>10){
      return system.getResultFail(-108, "尼斯小类不能超过10项");
    }
    var nclOneCodes2 = obj.nclOneCodes;
    for(var i=0;i<othertm.length;i++){//判断重复大类
      var other = othertm[i];
      if(other.nclOneCodes==obj.nclOneCodes){
        return system.getResultFail(-109, "该商标存在重复的尼斯大类");
      }else{
        nclOneCodes2=nclOneCodes2+","+other.nclOneCodes;
      }
    }
    return await self.db.transaction(async function (t) {
      var tmObj={
        id:tm.id,
        nclOneCodes:obj.nclOneCodes,
        nclSmallCodes: JSON.stringify(obj.nclSmallCodes),
        updateuser_id:user.id,
        updateuser:user.nickname
      };
      await self.dao.update(tmObj,t);//修改商标尼斯信息
      var otpObj={
        id:ordertmproduct.id,
        nclOneCodes:nclOneCodes2,
        updateuser_id:user.id,
        updateuser:user.nickname
      };
      await self.ordertmproductSve.dao.update(otpObj,t);//商标交付单 修改大类列表
      return system.getResultSuccess();
    })
  }
}
module.exports=TradeMarkService;
