const system = require("../../system");
const uuidv4 = require('uuid/v4');
const md5 = require("MD5");
var settings = require("../../../config/settings");

class OpPlatformUtils {
    constructor() {
        this.logCtl = system.getObject("web.common.oplogCtl");
        this.cacheManager = system.getObject("db.common.cacheManager");
        this.restClient = system.getObject("util.restClient");
        this.createUserUrl = settings.paasUrl() + "api/auth/accessAuth/register";
        this.fetchDefaultVCodeUrl = settings.paasUrl() + "api/auth/accessAuth/fetchDefaultVCode";
        this.loginUrl = settings.paasUrl() + "api/auth/accessAuth/login";
        this.authByCodeUrl = settings.paasUrl() + "api/auth/accessAuth/authByCode";

    }
    getUUID() {
        var uuid = uuidv4();
        var u = uuid.replace(/\-/g, "");
        return u;
    }

    async getReqApiAccessKey() {
        return await this.cacheManager["ApiAccessKeyCache"].cache(settings.appKey, null, 3600);
    }
    /**
     * 创建用户信息
     * @param {*} userName 用户名
     * @param {*} mobile 手机号
     * @param {*} password 密码，不传为使用默认密码
     * 
     * 返回值：
     * {
            "status": 0,---值为2000为已经存在此用户，注册失败
            "msg": "success",
            "data": {
                "auth_url": "http://sj.app.com:3002/auth?opencode=1e4949d1c39444a8b32f023143625b1d",---回调url，通过回调地址获取平台用户信息
                "opencode": "1e4949d1c39444a8b32f023143625b1d",---平台用户code随机生成会变，平台是30s有效期，通过其可以向获取用户信息
                "open_user_id": 12---平台用户id
            },
            "requestid": "5362bf6f941e4f92961a61068f05cd7f"
        }
     */
    async createUserInfo(userName, mobile, password) {
        var reqApiAccessKey = await this.getReqApiAccessKey();
        if (!reqApiAccessKey || !reqApiAccessKey.accessKey) {
            return system.getResult(null, "获取请求token失败");
        }
        var param = {
            userName: userName,
            mobile: mobile,
            password: password || settings.defaultPassWord,

        }
        //按照访问token
        var restResult = await this.restClient.execPostWithAK(
            param,
            this.createUserUrl, reqApiAccessKey.accessKey);

        if (restResult.status != 0 || !restResult.data) {
            return system.getResult(restResult.status, restResult.msg);
        }
        return system.getResultSuccess(restResult.data);
    }
    async fetchVCode(mobile) {
        var reqApiAccessKey = await this.getReqApiAccessKey();
        if (!reqApiAccessKey || !reqApiAccessKey.accessKey) {
            return system.getResult(null, "获取请求token失败");
        }
        var param = {
            mobile: mobile
        }
        //按照访问token
        var restResult = await this.restClient.execPostWithAK(
            param,
            this.fetchDefaultVCodeUrl, reqApiAccessKey.accessKey);

        if (restResult.status != 0 || !restResult.data) {
            return system.getResult(null, restResult.msg);
        }
        return system.getResultSuccess();
    }
    /**
     * 用户登录
     * @param {*} userName 用户名
     * @param {*} password 密码，不传为使用默认密码
     * 
     * 返回值：
     * {
            "status": 0,---值为2010为用户名或密码错误
            "msg": "success",
            "data": {
                "auth_url": "http://sj.app.com:3002/auth?opencode=1e4949d1c39444a8b32f023143625b1d",---回调url，通过回调地址获取平台用户信息
                "opencode": "1e4949d1c39444a8b32f023143625b1d"---平台用户code随机生成会变，平台是30s有效期，通过其可以向获取用户信息
            },
            "requestid": "5362bf6f941e4f92961a61068f05cd7f"
        }
     */
    async login(userName, password) {
        var reqApiAccessKey = await this.getReqApiAccessKey();
        if (!reqApiAccessKey || !reqApiAccessKey.accessKey) {
            return system.getResult(null, "获取请求token失败");
        }
        var param = {
            userName: userName,
            password: password || settings.defaultPassWord,

        }
        //按照访问token
        var restResult = await this.restClient.execPostWithAK(
            param,
            this.loginUrl, reqApiAccessKey.accessKey);

        if (restResult.status != 0 || !restResult.data) {
            return system.getResultFail(-1, restResult.msg, restResult.status);
        }
        return system.getResultSuccess(restResult.data);
    }
    /**
     * 通过opencode获取用户登录信息
     * @param {*} opencode 用户登录或注册opencode
     * 
     * 返回值：
     * {
            "status": 0,---值为2010为用户名或密码错误
            "msg": "success",
            "data": {},---平台用户信息
            "requestid": "5362bf6f941e4f92961a61068f05cd7f"
        }
    */
    async authByCode(opencode) {
        var reqApiAccessKey = await this.getReqApiAccessKey();
        if (!reqApiAccessKey || !reqApiAccessKey.accessKey) {
            return system.getResult(null, "获取请求token失败");
        }
        var param = {
            opencode: opencode
        }
        //按照访问token
        var restResult = await this.restClient.execPostWithAK(
            param,
            this.authByCodeUrl, reqApiAccessKey.accessKey);

        if (restResult.status != 0 || !restResult.data) {
            return system.getResult(restResult.status, restResult.msg);
        }
        return system.getResultSuccess(restResult.data);
    }

    /**
     * 登录并获取登录用户信息
     * @param {*} userName 
     * @param {*} password 
     */
    async loginInfo(userName, password) {
        var loginrs = await this.login(userName, password);
        if (loginrs.status !== 0) {
            return loginrs;
        }
        return await this.authByCode(loginrs.data.opencode)
    }
}

module.exports = OpPlatformUtils;