const system = require("../../../system");
const ServiceBase = require("../../sve.base")

/**
 * 订单产品表
 */
class OorderstatusService extends ServiceBase {
  constructor() {
    super("order", ServiceBase.getDaoName(OorderstatusService));

    this.oorderSve = system.getObject("service.order.oorderSve");

    this.oorderinforegDao = system.getObject("db.order.oorderinforegDao");
    this.oorderdeliverDao = system.getObject("db.order.oorderdeliverDao");
    this.oorderprocessDao = system.getObject("db.order.oorderprocessDao");
    this.obusinessmenDao = system.getObject("db.order.obusinessmenDao");

    this.osourceDao = system.getObject("db.common.osourceDao");
    this.oproductDao = system.getObject("db.product.oproductDao");
    this.oprocessDao = system.getObject("db.product.oprocessDao");
    this.oproductprocessDao = system.getObject("db.product.oproductprocessDao");
    this.BUSI_TYPE = 1; //个体公商户类型
    this.ISBANK = true; //银行开户
    this.TAXTYPE = [1, 2]; //个税和增值税类别集合
    this.PALTFORM_DELIVER_AUDIT_SUCCESS_CODE = "1160"; //平台交付审核成功状态码
    this.PALTFORM_DELIVER_AUDIT_FILE_CODE = "1150";    //平台交付审核失败状态码
  }

  /***************************************以下是订单处理的分发器***************************************************************** */

  /**
   * 验证订单状态
   * @param params
   * @returns {Promise<{msg: string, data: (*|null), bizmsg: string, status: number}>}
   * @id   订单ID
   * @status 订单下一个状态
   * @other  其他参数
   */
  async handleStatus(params) {
    try {
      let _order = await this.dao.findById(params.id);
      if (!_order) {
        return system.getResult(null, `订单不存在`);
      }

      // 完善订单特殊处理
      if (_order.status == 1010) {
        let chooseItems = [];
        for (let citem of this.trim(params.service_items).split(",")) {
          let productId = Number(this.trim(citem) || 0);
          if (productId) {
            chooseItems.push(productId);
          }
        }
        let orderProcessList = await this.oorderSve.buildOrderProcess(_order.product_id, chooseItems);
        if (!orderProcessList || orderProcessList.length == 0) {
          return system.getResult(null, "产品流程未配置");
        }

        for (let process of orderProcessList) {
          process.order_id = _order.id;
        }
        await this.oorderprocessDao.delByOrderId(_order.id);
        await this.oorderprocessDao.bulkCreate(orderProcessList);
      }

      let _orderStatus = await this.getOrderProcessStatus(_order.id, _order.status);
      let nextList = JSON.parse(_orderStatus.next_status);
      _orderStatus.next_status = nextList;
      let flag = false;
      let nextNames = [];
      let nextName = "";
      for (var nextItem of nextList) {
        nextNames.push(nextItem.next_name);
        if (nextItem.next_status == params.status) {
          flag = true;
          nextName = nextItem.next_name;
          break;
        }
      }
      if (!flag) {
        nextNames = nextNames.join("或");
        return system.getResult(null, `订单状态错误，下一订单状态应该是${nextNames}`);
      }

      params._order = _order;
      params._orderStatus = _orderStatus;
      params.nextName = nextName;
      //调用对应的方法 TODO SOMETHING .....

      // 调用func
      return await this[_orderStatus.func](params);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }

  }


  /**
   * 获取订单状态对象
   * @param order_id
   * @param status
   * @returns {Promise<*|{}>}
   */
  async getOrderProcessStatus(order_id, status) {
    let map = await this.oorderprocessDao.mapByOrderIdsAndStatus([order_id], [status]);
    return map[order_id + "_" + status] || {};
  }

  /*******************************以下是订单处理进度（平台）**************************************************** */


  /**
   * 分配业务员(平台)
   * @param {*} params
   * @id     String  订单ID
   * @bd_id  String  业务员ID
   * @status String 下一个状态码
   * @bd_path String 业务员权限
   * @assign_user_id String 分配人ID
   */
  async platformAassignSalesman(params) {
    if (!params.bd_id) {
      return system.getResult(null, `参数错误 业务员ID不能为空`);
    }
    try {
      //更新订单业务员
      let _order = params._order;
      _order.bd_id = this.trim(params.bd_id);
      _order.status = this.trim(params.status);
      _order.bd_path = this.trim(params.bd_path);
      _order.assign_time = new Date();
      _order.assign_user_id = this.trim(params.assign_user_id);
      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }


  /**
   * 业务员完善订单信息
   * @param {*} params
   * @id  String 订单ID
   * @status String 下一个状态 
   * @other Object 其他参数 
   */
  async perfectInformation(params) {
    //验证参数
    this.verificationPerfectInformation(params);
    let _oorderinforeg = await this.oorderinforegDao.findById(params._order.id);
    if (!_oorderinforeg) {
      return system.getResult(null, `订单异常 订单明细不存在，请联系管理员`);
    }

    //更新订单子表
    _oorderinforeg.legal_name = this.trim(params.legal_name);
    _oorderinforeg.id_card = this.trim(params.id_card);
    _oorderinforeg.legal_mobile = this.trim(params.legal_mobile);
    _oorderinforeg.names = this.trim(params.names);
    _oorderinforeg.capital = this.trim(params.capital);
    _oorderinforeg.domicile_id = this.trim(params.domicile_id);
    _oorderinforeg.domicile_name = this.trim(params.domicile_name);
    _oorderinforeg.business_scope_id = this.trim(params.business_scope_id);
    _oorderinforeg.business_type = this.trim(params.business_type);
    _oorderinforeg.business_scope = this.trim(params.business_scope);
    _oorderinforeg.idcard_front = this.trim(params.idcard_front);
    _oorderinforeg.idcard_back = this.trim(params.idcard_back);
    _oorderinforeg.other_file = this.trim(params.other_file);
    _oorderinforeg.guest_mail_addr = this.trim(params.guest_mail_addr);
    _oorderinforeg.guest_mail_to = this.trim(params.guest_mail_to);
    _oorderinforeg.guest_mail_mobile = this.trim(params.guest_mail_mobile);
    //更新主表
    params._order.service_items = this.trim(params.service_items);
    params._order.service_remark = this.trim(params.service_remark);
    params._order.status = this.trim(params.status);
    try {
      //开启事务
      await this.db.transaction(async t => {
        await _oorderinforeg.save();
        await params._order.save();
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  verificationPerfectInformation(params) {
    if (!params.legal_name) {
      return system.getResult(null, `参数错误 法人姓名不能为空`);
    }
    if (!params.id_card) {
      return system.getResult(null, `参数错误 法人身份证不能为空`);
    }
    if (!params.legal_mobile) {
      return system.getResult(null, `参数错误 法人电话不能为空`);
    }
    if (!params.names) {
      return system.getResult(null, `参数错误 个体户名称不能为空`);
    }
    if (!params.capital) {
      return system.getResult(null, `参数错误 注册资本不能为空`);
    }
    if (!params.domicile_id) {
      return system.getResult(null, `参数错误 注册地id不能为空`);
    }
    if (!params.domicile_name) {
      return system.getResult(null, `参数错误 注册地名称不能为空`);
    }
    if (!params.business_scope_id) {
      return system.getResult(null, `参数错误 经营范围id不能为空`);
    }
    if (!params.business_type) {
      return system.getResult(null, `参数错误 经营范围不能为空`);
    }
    if (!params.business_scope) {
      return system.getResult(null, `参数错误 经营范围详情不能为空`);
    }
    if (!params.idcard_front) {
      return system.getResult(null, `参数错误 身份证正面照片不能为空`);
    }
    if (!params.idcard_back) {
      return system.getResult(null, `参数错误 身份证反面照片不能为空`);
    }
    if (!params.other_file) {
      return system.getResult(null, `参数错误 其他文件不能为空`);
    }
    if (!params.guest_mail_addr) {
      return system.getResult(null, `参数错误 邮寄客户地址不能为空`);
    }
    if (!params.guest_mail_to) {
      return system.getResult(null, `参数错误 客户收件人不能为空`);
    }
    if (!params.guest_mail_mobile) {
      return system.getResult(null, `参数错误 客户收件人电话不能为空`);
    }

    // if(!params.guest_mail_no){
    //   return system.getResult(null,`参数错误 客户快递单号不能为空`);
    // }
    // if(!params.guest_mail_img){
    //   return system.getResult(null,`参数错误 交付商交付快递单号图片不能为空`);
    // }
    // if(!params.guest_accpet_file){
    //   return system.getResult(null,`参数错误 客户验收文件不能为空`);
    // }

    if (!params.service_items) {
      return system.getResult(null, `参数错误 服务项不能为空`);
    }

  }

  /**
   * 分配交付商
   * @param {*} params
   * @id  String 订单ID
   * @deliver_id String 交付商ID
   * @deliver_name String 交付商名称
   * @deliver_deliver String 交付商分成
   * @status String 下一个状态码 
   */
  async assignDeliver(params) {
    //参数验证
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.deliver_id) {
      return system.getResult(null, `参数错误 交付商ID不能为空`);
    }
    if (!params.hasOwnProperty(`deliver_divide`)) {
      return system.getResult(null, `参数错误 交付商分成不能为空`);
    }

    try {
      let self = this;
      await this.db.transaction(async t => {
        //创建orderdeliver记录
        let _oorderdeliver = await self.oorderdeliverDao.create({
          order_id: self.trim(params.id),
          deliver_id: self.trim(params.deliver_id),
          deliver_name: self.trim(params.deliver_name),
          deliver_divide: self.trim(params.deliver_divide),
        }, t);

        //更新oorder订单记录
        await self.dao.update({
          deliver_id: _oorderdeliver.id,
          id: self.trim(params.id),
          status: self.trim(params.status)
        }, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 平台交付审核（审核成功OR失败）
   * @param {*} params 
   * @id String 订单ID
   * 
   */
  async platformDeliverAudit(params) {
    //参数验证
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 订单状态不能为空`);
    }
    let oorderDeliverProperty = {};
    if (this.trim(params.status) == this.PALTFORM_DELIVER_AUDIT_SUCCESS_CODE) { //成功
      if (!params.audit_remark) {
        return system.getResult(null, `参数错误 交付审核备注不能为空`);
      }
      if (!params.deliver_content) {
        return system.getResult(null, `参数错误 交付内容注不能为空`);
      }
      if (!params.deliver_mail_addr) {
        return system.getResult(null, `参数错误 交付地址不能为空`);
      }
      if (!params.deliver_mail_to) {
        return system.getResult(null, `参数错误 收件人不能为空`);
      }
      if (!params.deliver_mail_mobile) {
        return system.getResult(null, `参数错误 联系电话不能为空`);
      }
      oorderDeliverProperty.audit_remark = this.trim(params.audit_remark);
      oorderDeliverProperty.deliver_content = this.trim(params.deliver_content);
      oorderDeliverProperty.deliver_mail_addr = this.trim(params.deliver_mail_addr);
      oorderDeliverProperty.deliver_mail_to = this.trim(params.deliver_mail_to);
      oorderDeliverProperty.deliver_mail_mobile = this.trim(params.deliver_mail_mobile);

    } else if (this.trim(params.status) == this.PALTFORM_DELIVER_AUDIT_FILE_CODE) {//失败
      // todo somthing...

      if (!this.trim(params.audit_remark)) {
        return system.getResult(null, `参数错误 交付审核备注不能为空`);
      }
      oorderDeliverProperty.audit_remark = this.trim(params.audit_remark);
    } else {
      return system.getResult(null, `参数错误 非法的订单状态`);
    }
    oorderDeliverProperty.id = params._order.deliver_id;
    oorderDeliverProperty.audit_result = params.nextName;
    let orderProperty = {};
    orderProperty.id = params.id;
    orderProperty.status = this.trim(params.status);

    try {
      let self = this;
      await this.db.transaction(async t => {
        //更新交付orderdeliver记录
        await self.oorderdeliverDao.update(oorderDeliverProperty, t);
        //更新oorder订单记录
        await self.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }


  /**
   * 平台验收
   * @param {*} params 
   */
  async platformAuditDetermine(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 订单状态不能为空`);
    }
    
    let orderDeliverProperty={};
    orderDeliverProperty.completed_at=new Date();
    orderDeliverProperty.id=params._order.deliver_id;

    let orderProperty = {};
    orderProperty.status = this.trim(params.status);
    orderProperty.id = this.trim(params._order.id);

    try {
      await this.db.transaction(async t=>{
        await this.dao.update(orderProperty,t);
        await this.oorderdeliverDao.update(orderDeliverProperty,t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }


  /**
   * 客户交付
   * @param {*} params 
   */
  async deliverCustomer(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 订单状态不能为空`);
    }
    if (!params.guest_mail_no) {
      return system.getResult(null, `参数错误 客户快递单号不能为空`);
    }
    if (!params.guest_mail_img) {
      return system.getResult(null, `参数错误 交付商交付快递单号图片不能为空`);
    }
    let oorderInforegProperty = {};
    oorderInforegProperty.id = this.trim(params._order.id);
    oorderInforegProperty.guest_mail_no = this.trim(params.guest_mail_no);
    oorderInforegProperty.guest_mail_img = this.trim(params.guest_mail_img);
    let oorderProperty = {};
    oorderProperty.status = this.trim(params.status);
    oorderProperty.id = params._order.id;
    try {
      await this.db.transaction(async t => {
        await this.oorderinforegDao.update(oorderInforegProperty, t);
        await this.dao.update(oorderProperty, t);
      });

      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 订单完成
   * @param {*} params 
   */
  async orderComplete(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 订单状态不能为空`);
    }
    if (!params.guest_accpet_file) {
      return system.getResult(null, `参数错误 客户验收文件不能为空`);
    }

    let oorderInforegProperty = {};
    oorderInforegProperty.id = this.trim(params._order.id);
    oorderInforegProperty.guest_accpet_file = this.trim(params.guest_accpet_file);

    let oorderProperty = {};
    oorderProperty.status = this.trim(params.status);
    oorderProperty.id = params._order.id;

    try {
      await this.db.transaction(async t => {
        await this.oorderinforegDao.update(oorderInforegProperty, t);
        await this.dao.update(oorderProperty, t);
      });

      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }





  /**********************************以下是交付商订单业务处理******************************************************** */

  /**
   * 分配业务员(交付商)
   * @param {*} params
   * @id     String  订单ID
   * @operator_id  String  业务员ID
   * @status String 下一个状态码
   */
  async deliverAssignSalesman(params) {
    if (!params.operator_id) {
      return system.getResult(null, `参数错误 业务员ID不能为空`);
    }
    try {
      let _oorderdeliver = await this.oorderdeliverDao.model.findOne({
        where: {
          order_id: this.trim(params._order.id)
        }
      });
      if (!_oorderdeliver) {
        return system.getResult(null, `参数错误 订单交付信息不存在`);
      }
      let orderdeliverProperty = {};
      orderdeliverProperty.operator_id = this.trim(params.operator_id);
      orderdeliverProperty.id = _oorderdeliver.id;

      let orderProperty = {};
      orderProperty.status = this.trim(params.status);
      orderProperty.id = this.trim(params._order.id);
      await this.db.transaction(async t => {
        await this.oorderdeliverDao.update(orderdeliverProperty, t);
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 工商注册中 
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async businessRegister(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }
    let _order = params._order;
    _order.status = this.trim(params.status);
    try {
      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }

  }

  /**
   * 工商注册完成
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async businessRegisterComplete(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }

    if (!params.name) {
      return system.getResult(null, `参数错误 个体工商户名称不能为空`);
    }
    if (!params.credit_code) {
      return system.getResult(null, `参数错误 统一社会信用代码不能为空`);
    }
    if (!params.business_place) {
      return system.getResult(null, `参数错误 经营场所不能为空`);
    }
    if (!params.business_scope) {
      return system.getResult(null, `参数错误 经营范围不能为空`);
    }
    if (!params.reg_date) {
      return system.getResult(null, `参数错误 注册日期不能为空`);
    }
    // if (!params.business_img) {
    //   return system.getResult(null, `参数错误 执照照片不能为空`);
    // }
    // if (!params.business_gov_file) {
    //   return system.getResult(null, `参数错误 工商官方文件不能为空`);
    // }
    let _order = params._order;

    let _orderInfo = await this.oorderinforegDao.findById(_order.id);

    let _obusinessmenProperty = {};
    _obusinessmenProperty.name = this.trim(params.name);
    _obusinessmenProperty.credit_code = this.trim(params.credit_code);
    _obusinessmenProperty.business_place = this.trim(params.business_place);
    _obusinessmenProperty.business_scope = this.trim(params.business_scope);
    _obusinessmenProperty.reg_date = this.trim(params.reg_date);
    _obusinessmenProperty.business_img = this.trim(params.business_img);
    _obusinessmenProperty.business_gov_file = this.trim(params.business_gov_file);
    _obusinessmenProperty.order_id = _order.id;
    _obusinessmenProperty.legal_name = _orderInfo.legal_name;
    _obusinessmenProperty.legal_mobile = _orderInfo.legal_mobile;
    _obusinessmenProperty.legal_idcard = _orderInfo.id_card;
    _obusinessmenProperty.bd_id = this.trim(params.bd_id);
    _obusinessmenProperty.bd_path = this.trim(params.bd_path);

    let orderProperty = {};
    orderProperty.id = _order.id;
    orderProperty.status = this.trim(params.status);
    orderProperty.busi_type = this.BUSI_TYPE;
    try {
      await this.db.transaction(async t => {
        let _obusinessmen = await this.obusinessmenDao.create(_obusinessmenProperty, t);
        orderProperty.busi_id = _obusinessmen.id;
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 刻章办理中
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async chapterEngraving(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }
    let _order = params._order;
    _order.status = this.trim(params.status);
    try {
      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(error);
    }
  }


  /**
   * 刻章办理已完成
   * @param {*} params 
   * @id  String 订单Id
   * @Statsu String 下一个订单
   * @other Object 其他参数
   * @gongzhang String 公章
   * @fapiaozhang String 发票章
   * @caiwuzhang String 财务章
   * @hetongzhang String 合同章
   * @farenzhang String 法人章
   * @zhang_gov_file String 刻章官方文件
   */
  async chapterEngravedComplete(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }

    // if (!params.gongzhang) {
    //   return system.getResult(null, `参数错误 公章不能为空`);
    // }
    // if (!params.fapiaozhang) {
    //   return system.getResult(null, `参数错误 发票章不能为空`);
    // }
    // if (!params.caiwuzhang) {
    //   return system.getResult(null, `参数错误 财务章不能为空`);
    // }
    // if (!params.hetongzhang) {
    //   return system.getResult(null, `参数错误 合同章不能为空`);
    // }
    // if (!params.farenzhang) {
    //   return system.getResult(null, `参数错误 法人章不能为空`);
    // }
    // if (!params.zhang_gov_file) {
    //   return system.getResult(null, `参数错误 刻章官方文件不能为空`);
    // }

    let _obusinessmenProperty = {};
    _obusinessmenProperty.gongzhang = this.trim(params.gongzhang);
    _obusinessmenProperty.fapiaozhang = this.trim(params.fapiaozhang);
    _obusinessmenProperty.caiwuzhang = this.trim(params.caiwuzhang);
    _obusinessmenProperty.hetongzhang = this.trim(params.hetongzhang);
    _obusinessmenProperty.farenzhang = this.trim(params.farenzhang);
    _obusinessmenProperty.zhang_gov_file = this.trim(params.zhang_gov_file);

    let _order = params._order;
    _obusinessmenProperty.id = _order.busi_id;
    let orderProperty = {};
    orderProperty.id = _order.id;
    orderProperty.status = this.trim(params.status);
    try {
      await this.db.transaction(async t => {
        await this.obusinessmenDao.update(_obusinessmenProperty, t);
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }


  /**
   * 银行开户中
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async backAccountOpening(params) {
    let _order = params._order;
    _order.status = this.trim(params.status);
    try {
      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(error);
    }
  }

  /**
   * 银行已开户
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async backAccountOpenComplete(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }

    if (!params.bank_name) {
      return system.getResult(null, `参数错误 账户名称不能为空`);
    }
    if (!params.bank_no) {
      return system.getResult(null, `参数错误 账户号不能为空`);
    }
    if (!params.bank) {
      return system.getResult(null, `参数错误 开户行不能为空`);
    }
    // if (!params.bank_img) {
    //   return system.getResult(null, `参数错误 账户信息不能为空`);
    // }
    // if (!params.bank_gov_file) {
    //   return system.getResult(null, `参数错误 银行开户官方文件不能为空`);
    // }

    let _obusinessmenProperty = {};
    _obusinessmenProperty.bank_name = this.trim(params.bank_name);
    _obusinessmenProperty.bank_no = this.trim(params.bank_no);
    _obusinessmenProperty.bank = this.trim(params.bank);
    _obusinessmenProperty.bank_img = this.trim(params.bank_img);
    _obusinessmenProperty.bank_gov_file = this.trim(params.bank_gov_file);

    let _order = params._order;
    _obusinessmenProperty.id = _order.busi_id;
    let orderProperty = {};
    orderProperty.id = _order.id;
    orderProperty.status = this.trim(params.status);
    try {
      await this.db.transaction(async t => {
        await this.obusinessmenDao.update(_obusinessmenProperty, t);
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 税务报道中
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async taxReporting(params) {
    let _order = params._order;
    _order.status = this.trim(params.status);
    try {
      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(error);
    }
  }

  /**
   * 税务报道完成
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async taxReportComplete(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }

    if (!params.tax_reg_day) {
      return system.getResult(null, `参数错误 税务登记日不能为空`);
    }
    if (!params.tax_org) {
      return system.getResult(null, `参数错误 税务机构名称不能为空`);
    }
    // if (!params.ca_img) {
    //   return system.getResult(null, `参数错误 CA照片不能为空`);
    // }
    // if (!params.tax_gov_file) {
    //   return system.getResult(null, `参数错误 税务报道官方文件不能为空`);
    // }

    let _obusinessmenProperty = {};
    _obusinessmenProperty.tax_reg_day = this.trim(params.tax_reg_day);
    _obusinessmenProperty.tax_org = this.trim(params.tax_org);
    _obusinessmenProperty.ca_img = this.trim(params.ca_img);
    _obusinessmenProperty.tax_gov_file = this.trim(params.tax_gov_file);

    let _order = params._order;
    _obusinessmenProperty.id = _order.busi_id;
    let orderProperty = {};
    orderProperty.id = _order.id;
    orderProperty.status = this.trim(params.status);
    try {
      await this.db.transaction(async t => {
        await this.obusinessmenDao.update(_obusinessmenProperty, t);
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 核定税种
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async taxCategoryDetermine(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }

    if (!params.common_tax_ladder) {
      return system.getResult(null, `参数错误 普票个税阶梯不能为空`);
    }
    if (!params.common_other_ladder) {
      return system.getResult(null, `参数错误 普票增值税、附加税阶梯不能为空`);
    }
    if (!params.special_tax_ladder) {
      return system.getResult(null, `参数错误 专票个税阶梯不能为空`);
    }
    if (!params.special_other_ladder) {
      return system.getResult(null, `参数错误 专票增值税、附加税阶梯不能为空`);
    }
    if (!params.add_value_up_type && this.TAXTYPE.indexOf(Number(params.add_value_up_type)) == -1) {
      return system.getResult(null, `参数错误 增值税累计类型不能为空`);
    }
    if (!params.tax_up_type && this.TAXTYPE.indexOf(Number(params.tax_up_type)) == -1) {
      return system.getResult(null, `参数错误 个税累计类型不能为空`);
    }

    let _obusinessmenProperty = {};
    _obusinessmenProperty.common_tax_ladder = JSON.stringify(params.common_tax_ladder);
    _obusinessmenProperty.common_other_ladder = JSON.stringify(params.common_other_ladder);
    _obusinessmenProperty.special_tax_ladder = JSON.stringify(params.special_tax_ladder);
    _obusinessmenProperty.special_other_ladder = JSON.stringify(params.special_other_ladder);

    _obusinessmenProperty.add_value_up_type = this.trim(params.add_value_up_type);
    _obusinessmenProperty.tax_up_type = this.trim(params.tax_up_type);

    let _order = params._order;
    _obusinessmenProperty.id = _order.busi_id;
    let orderProperty = {};
    orderProperty.id = _order.id;
    orderProperty.status = this.trim(params.status);
    try {
      await this.db.transaction(async t => {
        await this.obusinessmenDao.update(_obusinessmenProperty, t);
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 交付商提交审核
   * @param {*} params 
   * @id  订单ID 
   * @status String 下一个订单状态
   */
  async submitAudit(params) {
    let _order = params._order;
    _order.status = this.trim(params.status);
    try {
      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(error);
    }
  }

  /**
   * 交付商订单交付
   * @param {*} params
   * @id     String 订单ID
   * @status String 订单状态
   */
  async mailed(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.status) {
      return system.getResult(null, `参数错误 状态码不能为空`);
    }

    if (!params.deliver_mail_no) {
      return system.getResult(null, `参数错误 交付商交付快递单号不能为空`);
    }
    if (!params.deliver_mail_img) {
      return system.getResult(null, `参数错误 交付商交付快递单号图片不能为空`);
    }
    let oorderDeliverProperty = {};
    oorderDeliverProperty.deliver_mail_no = this.trim(params.deliver_mail_no);
    oorderDeliverProperty.deliver_mail_img = this.trim(params.deliver_mail_img);

    let _order = params._order;
    oorderDeliverProperty.id = _order.deliver_id;

    let orderProperty = {};
    orderProperty.status = this.trim(params.status);
    orderProperty.id = _order.id;

    try {
      await this.db.transaction(async t => {
        await this.oorderdeliverDao.update(oorderDeliverProperty, t);
        await this.dao.update(orderProperty, t);
      });
      return system.getResultSuccess();
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息${error}`);
    }
  }







}

module.exports = OorderstatusService;