const system = require("../../../system");
const ServiceBase = require("../../sve.base")
const moment = require("moment");
/**
 * 订单产品表
 */
class OorderService extends ServiceBase {
  constructor() {
    super("order", ServiceBase.getDaoName(OorderService));

    this.oorderinforegDao = system.getObject("db.order.oorderinforegDao");
    this.oorderdeliverDao = system.getObject("db.order.oorderdeliverDao");
    this.oorderprocessDao = system.getObject("db.order.oorderprocessDao");
    this.obusinessmenDao = system.getObject("db.order.obusinessmenDao");
    this.oorderdeliverDao = system.getObject("db.order.oorderdeliverDao");

    this.osourceDao = system.getObject("db.common.osourceDao");

    this.oproductDao = system.getObject("db.product.oproductDao");
    this.oprocessDao = system.getObject("db.product.oprocessDao");
    this.oproductprocessDao = system.getObject("db.product.oproductprocessDao");
  }

  /**
   *  新增渠道订单
   * @returns {Promise<void>}
   */
  async addSourceOrder(params) {
    // 整理参数
    let order = {
      merchant_id: "",
      channel_order_no: "",
      busi_type: 1,
      busi_id: "",
      price: 0,
    };
    order.product_id = Number(params.product_id);
    order.source_id = Number(params.source_id);
    order.source_no = this.trim(params.source_no);
    order.desc = this.trim(params.notes);
    order.contact_mobile = this.trim(params.contact_mobile);

    // 验证订单是否存在
    let exists = await this.dao.findOne({ source_id: order.source_id, source_no: order.source_no });
    if (exists) {
      return system.getResult(null, `订单号${order.source_no}已存在`);
    }

    // 验证来源
    let source = await this.osourceDao.findById(order.source_id);
    if (!source) {
      return system.getResult(null, "来源错误");
    }

    // 验证产品
    let product = await this.oproductDao.findById(order.product_id);
    if (!product) {
      return system.getResult(null, "产品错误");
    }

    // 构建订单流程列表
    let orderProcessList = await this.buildOrderProcess(order.product_id);
    if (!orderProcessList || orderProcessList.length == 0) {
      return system.getResult(null, "产品流程未配置");
    }

    order.status = orderProcessList[0].status;
    // 开启事务
    var self = this;
    await this.db.transaction(async function (t) {
      // 插入订单数据
      order = await self.dao.create(order, t);

      // 插入订单状态数据
      for (let op of orderProcessList) {
        op.order_id = order.id;
      }
      await self.oorderprocessDao.bulkCreate(orderProcessList, t);

      let orderReg = {
        id: order.id,
        legal_mobile: order.contact_mobile,
      }
      await self.oorderinforegDao.create(orderReg, t);
      return order;
    });

    return this.getOrderProcessStatus(order.id, order.status);
    // return order;
  }

  /**
   * 查询订单信息
   * @param {*} params
   */
  async info(params) {
    let row = await this.dao.getById(params.id);
    if (!row) {
      return system.getResult(null, "订单不存在");
    }
    await this.dao.setRowCodeName(row, "status");
    let statusObj = await this.getOrderProcessStatus(row.id, row.status);

    row.next_status = JSON.parse(statusObj.next_status || "[]");
    row.status_name = statusObj.name;

    this.handleDate(row, ["created_at", "assign_time"], null, -8);
    return system.getResultSuccess(row);
  }

  async infoAll(params) {
    let result = {};
    let order = await this.dao.getById(params.id);
    let orderInfo = await this.oorderinforegDao.getById(params.id);
    let businessmen = {};
    if (order.busi_type == 1) {
      businessmen = await this.obusinessmenDao.getById(order.busi_id) || {};
    }
    let orderDeliver = await this.oorderdeliverDao.getById(order.deliver_id) || {};

    await this.formateStatus([order]);
    await this.formateSource([order]);
    await this.formateProduct([order]);

    this.handleDate(order, ["created_at", "sign_time", "assign_time"], "YYYY-MM-DD HH:mm:ss", -8);
    this.handleDate(businessmen, ["tax_reg_day", "created_at", 'reg_date'], "YYYY-MM-DD HH:mm:ss", -8);
    this.handleDate(orderInfo, ["created_at"], "YYYY-MM-DD HH:mm:ss", -8);

    orderDeliver.deliver_divide = system.f2y(orderDeliver.deliver_divide);
    orderInfo.capital = system.f2y(orderInfo.capital);
    order.service_items = order.service_items.split(",")
    result.order = order;
    result.regInfo = orderInfo;
    result.businessmen = businessmen;
    result.orderDeliver = orderDeliver;
    return system.getResultSuccess(result);
  }

  /**
   * 构建产品流程对象
   * @param productPid
   * @param chooseProductIds
   * @returns {Promise<void>}
   */
  async buildOrderProcess(productPid, chooseProductIds) {
    // 查询所有产品子项
    let productList = await this.oproductDao.findListByPid(productPid);

    // 根据所选产品id, 查询产品流程
    let productIds = [];
    for (let product of productList) {
      // 过滤未选择产品
      if (product.is_choose && chooseProductIds && chooseProductIds.indexOf(product.id) == -1) {
        continue;
      }
      productIds.push(product.id);
    }
    if (productIds.length == 0) {
      return null;
    }

    // 查询产品流程
    let productProcessMap = await this.oproductprocessDao.findMapByProductProductIds(productPid, productIds);

    // 批量查流程
    let processMap = await this.oprocessDao.mapAll();

    let orderProcessList = [];

    for (let productId of productIds) {
      // 产品子项流程列表
      let productProcessList = productProcessMap[productId];
      if (!productProcessList || productProcessList.length == 0) {
        continue;
      }

      for (let idx = 0; idx < productProcessList.length; idx++) {
        // 风还钻该处理每一个子项流程 变为 订单流程对象
        let productProcess = productProcessList[idx];
        let process = processMap[productProcess.process_id];
        let nextArr = this.trim(productProcess.next_status).split(",");
        let nextStatus = [];
        for (var nextId of nextArr) {
          nextId = Number(nextId || 0);
          let nextObj = processMap[nextId];
          if (!nextObj) {
            continue;
          }
          nextStatus.push({ next_status: nextObj.status, next_name: nextObj.name });
        }

        let orderProcess = {
          product_id: productPid,
          name: process.name,
          status: process.status,
          func: productProcess.func,
          next_status: JSON.stringify(nextStatus),
          name1: productProcess.name1,
          name2: productProcess.name2,
          name3: productProcess.name3,
          name4: productProcess.name4,
          sort: productProcess.sort,
          autoIncrement: true
        };
        // 上一个产品流程于当前产品流程连接
        if (idx == 0 && orderProcessList.length > 0) {
          let lastProcess = orderProcessList[orderProcessList.length - 1];
          let nextObj = processMap[productProcess.process_id];
          lastProcess.next_status = JSON.stringify([{ next_status: nextObj.status, next_name: nextObj.name }]);
        }
        orderProcessList.push(orderProcess);
      }
    }
    return orderProcessList;
  }

  /**
   * 订单管理列表
   * @param {*} params
   */
  async orders(params) {
    let where = {};
    if (params.id) {
      where.id = this.trim(params.id);
    }
    if (params.status) {
      where.status = this.trim(params.status);
    }
    if (params.deliver_id) {
      where.deliver_id = this.trim(params.deliver_id);
    }
    if (params.bd_path) {
      where.bd_path = {
        [this.db.Op.like]: this.trim(params.bd_path) + "%"
      }
    }

    var currentPage = Number(params.currentPage || 1);
    var pageSize = Number(params.pageSize || 10);
    var orderby = [
      ["id", 'desc']
    ];
    this.addWhereTime(where, 'created_at', params.createdBegin, params.createdEnd);

    try {
      var page = await this.getPageList(currentPage, pageSize, where, orderby, null);
      //格式化订单来源
      await this.formateSource(page.rows);
      //格式化订单产品
      await this.formateProduct(page.rows);
      //格式化交付商
      await this.formateDeliver(page.rows);
      //格式化业务员
      await this.formateStatus(page.rows);

      for (let item of page.rows) {
        this.handleDate(item, ["created_at", "sign_time"], "YYYY-MM-DD HH:mm:ss", -8);
      }

      return system.getResult(page);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 格式化列表来源数据
   * @param {*} orderList
   */
  async formateSource(orderList) {
    let sourceSet = new Set();
    let sourceIdList = [];
    for (let item of orderList) {
      if (item.source_id) {
        sourceSet.add(item.source_id);
      }
    }
    sourceIdList = Array.from(sourceSet);
    let sourceMap = await this.osourceDao.mapByIds(sourceIdList);
    for (let item of orderList) {
      item.osource = sourceMap[item.source_id] || {}
    }
  }

  /**
   * 格式化订单产品
   * @param {*} orderList
   */
  async formateProduct(orderList) {
    let productIdList = [];
    for (let item of orderList) {
      if (item.product_id) {
        productIdList.push(item.product_id);
      }
    }

    let productMap = await this.oproductDao.mapByIds(productIdList);
    for (let item of orderList) {
      item.oproduct = productMap[item.product_id] || {}
    }
  }

  /**
   * 格式化订单交付商
   * @param {*} orderList
   */
  async formateDeliver(orderList) {
    let deliverIdList = [];
    for (let item of orderList) {
      if (item.deliver_id) {
        deliverIdList.push(item.deliver_id);
      }
    }

    let deliverMap = await this.oorderdeliverDao.mapByIds(deliverIdList);
    for (let item of orderList) {
      item.odeliver = deliverMap[item.deliver_id] || {}
    }
  }

  /**
   * 处理订单状态数据
   * @param orderList
   * @returns {Promise<void>}
   */
  async formateStatus(orderList) {
    let ids = [];
    let statuses = [];
    for (let item of orderList) {
      ids.push(item.id);
      statuses.push(item.status);
    }

    let map = await this.oorderprocessDao.mapByOrderIdsAndStatus(ids, statuses);

    for (let item of orderList) {
      let key = item.id + "_" + item.status;
      let v = map[key] || {};
      item.statusName = v.name;
      item.status_name = v.name;
      item.next_status = JSON.parse(v.next_status || "[]");
    }
    console.log(orderList);
  }

  /**
   * 获取订单状态对象
   * @param order_id
   * @param status
   * @returns {Promise<*|{}>}
   */
  async getOrderProcessStatus(order_id, status) {
    let map = await this.oorderprocessDao.mapByOrderIdsAndStatus([order_id], [status]);
    return map[order_id + "_" + status] || {};
  }

  /**
   * 分配业务员
   * @param {*} params
   * @id     String  订单ID
   * @bd_id  String  业务员ID
   * @status String 下一个状态码
   * @bd_path String 业务员权限
   */
  async assignSalesman(params) {
    if (!params.bd_id) {
      return system.getResult(null, `参数错误 业务员ID不能为空`);
    }
    try {
      //更新订单业务员
      let _order = await this.findById(params.id);
      if (!_order) {
        return system.getResult(null, `订单不存在`);
      }
      //to do ... 验证下一个状态
      await this.statusAction(params);

      _order.bd_id = this.trim(params.bd_id);
      _order.status = this.trim(params.status);
      _order.bd_path = this.trim(params.bd_path);

      let res = await _order.save();
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 验证订单状态
   * @param params
   * @returns {Promise<{msg: string, data: (*|null), bizmsg: string, status: number}>}
   */
  async statusAction(params) {
    let _order = await this.findById(params.id);
    if (!_order) {
      return system.getResult(null, `订单不存在`);
    }
    //todo ...  验证状态
    let _orderStatus = await this.getOrderProcessStatus(_order.id, _order.status);
    let nextStatus = JSON.parse(_orderStatus.next_status);
    if (params.status != nextStatus.next_status) {
      return system.getResult(null, `订单状态错误，下一订单状态应该是${nextStatus.next_name}`);
    }
    params._order = _order;
    return await this['assignDeliver'](params);
  }


  /**
   * 分配交付商
   * @param {*} params
   * @id  String 订单ID
   * @deliver_id String 交付商ID
   * @deliver_name String 交付商名称
   * @deliver_deliver String 交付商分成
   * @status String 下一个状态码
   */
  async assignDeliver(params) {
    //参数验证
    if (!params.id) {
      return system.getResult(null, `参数错误 订单ID不能为空`);
    }
    if (!params.deliver_id) {
      return system.getResult(null, `参数错误 交付商ID不能为空`);
    }
    if (!params.hasOwnProperty(`deliver_divide`)) {
      return system.getResult(null, `参数错误 交付商分成不能为空`);
    }

    try {
      let self = this;
      let res = await this.db.transaction(async t => {
        //创建orderdeliver记录
        await self.oorderdeliverDao.create({
          order_id: self.trim(params.id),
          deliver_id: self.trim(params.deliver_id),
          deliver_name: self.trim(params.deliver_name),
          deliver_divide: self.trim(params.deliver_divide),
        }, t);

        //更新oorder订单记录
        await self.dao.update({
          deliver_id: self.trim(params.deliver_id),
          id: self.trim(params.id),
          status: self.trim(params.status)
        }, t);

      });
      return system.getResult(res);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  /**
   * 平台数据统计 (订单数量和交易额度)
   * @param {*} params 
   */
  async statTransData(params) {
    if (!params.current_date) {
      return system.getResult(null, `参数错误 当前时间不能为空`);
    }
    try {
      var result = {
        orderCount: 0,
        orderPrice: 0,
      };
      var begin, end;
      // 取开始时间和结束时间
      begin = moment(params.current_date).format("YYYY-MM") + "-01 00:00:00";
      end = moment(begin).add(1, "months").subtract(1, 'days').format("YYYY-MM-DD") + " 23:59:59";

      // echart数据  开始
      var days = [];
      var dayCounts = [];
      var priceCounts = [];

      // 处理查询业务
      days = this.getDays(end);
      // 按天统计
      var dayMap = await this.dao.statDayByTime(begin, end);
      for (var day of days) {
        var ditem = dayMap[day] || {};
        dayCounts.push(ditem.orderCount || 0);
        priceCounts.push(system.f2y(ditem.orderPrice || 0));
      }

      //echart数据 结束

      //当月订单的数量统计和订单总金额
      var orderData = await this.dao.statOrderByTime(begin, end) || {};
      result.orderCount = orderData[orderData.begin].orderCount;
      result.orderPrice = system.f2y(orderData[orderData.begin].orderPrice);

      result.days = days;
      result.dayCounts = dayCounts;
      result.priceCounts = priceCounts;
      return system.getResult(result);
    } catch (error) {
      console.log(error);
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }

  }

  getDays(end) {
    var days = [];
    var month = moment(end).format("YYYY-MM");
    var endDay = Number(moment(end).format("DD"));
    for (var i = 1; i <= endDay; i++) {
      if (i < 10) {
        days.push(month + "-0" + i);
      } else {
        days.push(month + "-" + i);
      }
    }
    return days;
  }

  /**
   * 待办事项
   * @param {*} params 
   */
  async needToBeDealtWith(params) {
    if (!params.current_date) {
      return system.getResult(null, `参数错误 当前时间不能为空`);
    }
    try {
      let begin, end;
      begin = moment(params.current_date).format("YYYY-MM") + "-01 00:00:00";
      end = moment(begin).add(1, "months").subtract(1, 'days').format("YYYY-MM-DD") + " 23:59:59";
      //已经完成
      let orderComplete = await this.dao.model.count({
        where: {
          created_at : {
            [this.db.Op.between]: [begin, end]
          },
          status:"1200"
        }
      });
      //待分配
      let orderUnassignment = await this.dao.model.count({
        where: {
          created_at : {
            [this.db.Op.between]: [begin, end]
          },
          status:"1000"
        }
      });
      //待审核
      let orderUnexamine = await this.dao.model.count({
        where: {
          created_at : {
            [this.db.Op.between]: [begin, end]
          },
          status:"1140"
        }
      });
      //办理中
      let orderHandling = await this.dao.model.count({
        where: {
          created_at : {
            [this.db.Op.between]: [begin, end]
          },
          status:{
            [this.db.Op.between]: ["1030", '1130']
          }
        }
      });
      return system.getResult({
        orderComplete,orderUnassignment,orderUnexamine,orderHandling
      });
    } catch (error) {
      console.log(error);
      return system.getResult(null, error);
    }
  }

  /**
   * 交付商业务概览
   * @param {*} params 
   */
  async deliverData(params){
    if(!params.current_date){
      return system.getResult(null,`参数错误 当前时间不能为空`);
    }
    try {
      let begin, end;
      begin = moment(params.current_date).format("YYYY-MM") + "-01 00:00:00";
      end = moment(begin).add(1, "months").subtract(1, 'days').format("YYYY-MM-DD") + " 23:59:59";
      let deliverDataList = await this.dao.deliverData(begin,end);
      return system.getResult(deliverDataList);
    } catch (error) {
      console.log(error);
      return system.getResult(error);
    }
  }
}

module.exports = OorderService;