const ServiceBase = require("../../sve.base");
const system = require("../../../system");
const moment = require('moment')
/**
 * 交付商 提交的信息
 */
class InvoiceService extends ServiceBase {
  constructor() {
    super("invoice", ServiceBase.getDaoName(InvoiceService));
    let is = system.getObject("util.invoiceStatus");
    this.invoiceStatus = is.status;

    this.applyDao = system.getObject("db.invoice.applyDao");
    this.delivererDao = system.getObject("db.invoice.delivererDao");
  }

  /**
   * 完税证明
   * @param {*} params 
   */
  async apiTxPayment(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 发票ID不合法`);
    }
    try {
      let res = await this.txPayment(params);
      return res;
    } catch (error) {
      return system.getResult(null, `参数错误 错误信息 ${error}`);
    }
  }

  /**
   * 查看完税证明
   * @param {*} params 
   */
  async apiQueryTxPayment(params) {
    if (!params.id) {
      return system.getResult(null, `参数错误 发票ID不合法`);
    }
    try {
      return await this.queryTxPayment(params);
    } catch (error) {
      return system.getResult(null, `参数错误 错误信息 ${error}`);
    }
  }

  /**
   * 发票列表（平台）
   * @param {*} params 
   */
  async apiQueryInvoices(params) {
    try {
      if(params.applyNo){params.applyNo=this.trim(params.applyNo)}
      if(params.type){params.type=this.trim(params.type);}
      if(params.invoiceTime){params.invoiceTime=this.trim(params.invoiceTime);}
      return await this.queryInvoices(params);
    } catch (error) {
      return system.getResult(null, `系统错误 错误信息 ${error}`);
    }
  }

  //==============================================================

  /**
   * 完税证明
   * @param {*} params
   * {
   *    id:xxx   //完税证明
   * } 
   */
  async txPayment(params) {
    try {
      let _invoice = await this.dao.findOne({
        id: this.trim(params.id)
      });
      if (!_invoice) {
        return system.getResult(null, `发票不存在`);
      }
      let res = await this.dao.update({
        id: this.trim(params.id),
        taxNo: this.trim(params.taxNo),
        complateTax: 1,
        taxTime: this.trim(params.taxTime),
        taxVoucher: this.trim(params.taxVoucher)
      });
      return system.getResult(res);
    } catch (error) {
      return system.getResult(null, `参数错误 错误信息 ${error}`);
    }
  }

  /**
   * 查看完税证明
   * @param {*} params
   * {
   *    id:xxx   //完税证明
   * } 
   */
  async queryTxPayment(params) {
    try {
      let _invoice = await this.dao.model.findOne({
        where: {
          id: this.trim(params.id),
          complateTax: 1
        },
        attributes: ['id', 'taxNo', 'complateTax', 'taxTime', 'taxVoucher']
      });
      if (!_invoice) {
        return system.getResult(null, `完税证明不存在`);
      }
      return system.getResult(_invoice);
    } catch (error) {
      return system.getResult(null, `参数错误 错误信息 ${error}`);
    }
  }



  /**
   * 发票列表查询(平台)
   * @param {*} params 
   * @param applyNo   发票的申请编号
   * @param invoiceTime 发票的申请时间
   * @param type      发票的类型
   * @param status    业务进度
   */
  async queryInvoices(params) {
    try {

      var pageIndex = Number(params.pageIndex || 1);
      var pageSize = Number(params.pageSize || 10);

      var total = await this.dao.countByParams(params);
      if (total == 0) {
        return system.getResultSuccess({
          count: 0,
          rows: []
        });
      }

      var startRow = (pageIndex - 1) * pageSize;

      var list = await this.dao.pageByParams(params, startRow, pageSize);

      await this.setApply(list);
      for (var item of list) {
        // item.redStatusName = this.dao.getRowCodeName(item, "red_status");

        if (item.red_status == 1) {
          item.red_status = '未红冲';
        } else if (item.red_status == 2) {
          item.red_status = '红冲中';
        } else if (item.red_status == 3) {
          item.red_status = '红冲失败'
        } else if (item.red_status == 4) {
          item.red_status = '红冲成功';
        } else {
          item.red_status = '';
        }
        this.dao.getRowCodeName(item.apply,"status");
        item.complate_tax = item.complate_tax == 1 ? '已完税' : '未完税';
        item.month = moment(item.invoice_time).month() + 1;
        this.handleDate(item,['invoice_time'],null,-8);
        if (item.apply['type'] == '10') {
          item.apply['type'] = "普通发票";
        } else if (item.apply['type'] == '20') {
          item.apply['type'] = "增值税专用发票";
        } else {
          item.apply['type'] = "电子发票";
        }
      }
      var page = {
        count: total,
        rows: list
      };
      return system.getResultSuccess(page);
    } catch (error) {
      console.log(error);
      return system.getResult(`系统错误 错误信息 ${error}`);
    }
  }

  async setApply(list) {
    if (!list || list.length == 0) {
      return;
    }

    var ids = [];
    for (var item of list) {
      ids.push(item.id);
    }

    var applyMap = await this.applyDao.findMapByIds(ids) || {};
    for (var item of list) {
      item.apply = applyMap[item.id] || {};
    }
  }

}
module.exports = InvoiceService;