'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; /**
                                                                                                                                                                                                                                                                   *  Copyright (c) Facebook, Inc.
                                                                                                                                                                                                                                                                   *  All rights reserved.
                                                                                                                                                                                                                                                                   *
                                                                                                                                                                                                                                                                   *  This source code is licensed under the license found in the
                                                                                                                                                                                                                                                                   *  LICENSE file in the root directory of this source tree.
                                                                                                                                                                                                                                                                   *
                                                                                                                                                                                                                                                                   *  
                                                                                                                                                                                                                                                                   */

exports.getOutline = getOutline;

var _graphql = require('graphql');

var _graphqlLanguageServiceUtils = require('@apollographql/graphql-language-service-utils');

var INLINE_FRAGMENT = _graphql.Kind.INLINE_FRAGMENT;


var OUTLINEABLE_KINDS = {
  Field: true,
  OperationDefinition: true,
  Document: true,
  SelectionSet: true,
  Name: true,
  FragmentDefinition: true,
  FragmentSpread: true,
  InlineFragment: true
};

function getOutline(queryText) {
  var ast = void 0;
  try {
    ast = (0, _graphql.parse)(queryText);
  } catch (error) {
    return null;
  }

  var visitorFns = outlineTreeConverter(queryText);
  var outlineTrees = (0, _graphql.visit)(ast, {
    leave: function leave(node) {
      if (OUTLINEABLE_KINDS.hasOwnProperty(node.kind) && visitorFns[node.kind]) {
        return visitorFns[node.kind](node);
      }
      return null;
    }
  });
  return { outlineTrees: outlineTrees };
}

function outlineTreeConverter(docText) {
  var meta = function meta(node) {
    return {
      representativeName: node.name,
      startPosition: (0, _graphqlLanguageServiceUtils.offsetToPosition)(docText, node.loc.start),
      endPosition: (0, _graphqlLanguageServiceUtils.offsetToPosition)(docText, node.loc.end),
      children: node.selectionSet || []
    };
  };
  return {
    Field: function Field(node) {
      var tokenizedText = node.alias ? [buildToken('plain', node.alias), buildToken('plain', ': ')] : [];
      tokenizedText.push(buildToken('plain', node.name));
      return _extends({ tokenizedText: tokenizedText }, meta(node));
    },
    OperationDefinition: function OperationDefinition(node) {
      return _extends({
        tokenizedText: [buildToken('keyword', node.operation), buildToken('whitespace', ' '), buildToken('class-name', node.name)]
      }, meta(node));
    },
    Document: function Document(node) {
      return node.definitions;
    },
    SelectionSet: function SelectionSet(node) {
      return concatMap(node.selections, function (child) {
        return child.kind === INLINE_FRAGMENT ? child.selectionSet : child;
      });
    },
    Name: function Name(node) {
      return node.value;
    },
    FragmentDefinition: function FragmentDefinition(node) {
      return _extends({
        tokenizedText: [buildToken('keyword', 'fragment'), buildToken('whitespace', ' '), buildToken('class-name', node.name)]
      }, meta(node));
    },
    FragmentSpread: function FragmentSpread(node) {
      return _extends({
        tokenizedText: [buildToken('plain', '...'), buildToken('class-name', node.name)]
      }, meta(node));
    },
    InlineFragment: function InlineFragment(node) {
      return node.selectionSet;
    }
  };
}

function buildToken(kind, value) {
  return { kind: kind, value: value };
}

function concatMap(arr, fn) {
  var res = [];
  for (var i = 0; i < arr.length; i++) {
    var x = fn(arr[i], i);
    if (Array.isArray(x)) {
      res.push.apply(res, x);
    } else {
      res.push(x);
    }
  }
  return res;
}